<?php
/**
 * Sistema de Log de Erros - TRDAP
 * Captura e registra todos os erros PHP incluindo erros 500
 */

class ErrorHandler {
    private static $logDir = '';
    private static $logFile = '';
    
    public static function init() {
        self::$logDir = __DIR__;
        self::$logFile = self::$logDir . '/error.log';
        
        // Garantir que o diretório de logs existe
        if (!file_exists(self::$logDir)) {
            mkdir(self::$logDir, 0755, true);
        }
        
        // Configurar tratamento de erros
        set_error_handler([self::class, 'handleError']);
        set_exception_handler([self::class, 'handleException']);
        register_shutdown_function([self::class, 'handleFatalError']);
        
        // Configurações do PHP para logging
        ini_set('log_errors', 1);
        ini_set('error_log', self::$logFile);
        
        // Reportar todos os erros para o log
        error_reporting(E_ALL);
        
        // Não exibir erros no navegador (segurança)
        ini_set('display_errors', 0);
        ini_set('display_startup_errors', 0);
    }
    
    /**
     * Manipula erros PHP regulares
     */
    public static function handleError($errno, $errstr, $errfile, $errline) {
        $errorTypes = [
            E_ERROR             => 'ERROR',
            E_WARNING           => 'WARNING',
            E_PARSE             => 'PARSE',
            E_NOTICE            => 'NOTICE',
            E_CORE_ERROR        => 'CORE_ERROR',
            E_CORE_WARNING      => 'CORE_WARNING',
            E_COMPILE_ERROR     => 'COMPILE_ERROR',
            E_COMPILE_WARNING   => 'COMPILE_WARNING',
            E_USER_ERROR        => 'USER_ERROR',
            E_USER_WARNING      => 'USER_WARNING',
            E_USER_NOTICE       => 'USER_NOTICE',
            E_STRICT            => 'STRICT',
            E_RECOVERABLE_ERROR => 'RECOVERABLE_ERROR',
            E_DEPRECATED        => 'DEPRECATED',
            E_USER_DEPRECATED   => 'USER_DEPRECATED',
        ];
        
        $errorType = isset($errorTypes[$errno]) ? $errorTypes[$errno] : 'UNKNOWN';
        
        $message = sprintf(
            "[%s] [%s] %s in %s on line %d",
            date('Y-m-d H:i:s'),
            $errorType,
            $errstr,
            $errfile,
            $errline
        );
        
        self::logError($message);
        
        // Para erros fatais, exibir página de erro amigável
        if (in_array($errno, [E_ERROR, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR, E_RECOVERABLE_ERROR])) {
            self::showErrorPage();
            exit(1);
        }
        
        return false;
    }
    
    /**
     * Manipula exceções não capturadas
     */
    public static function handleException($exception) {
        $message = sprintf(
            "[%s] [EXCEPTION] %s: %s in %s on line %d\nStack trace:\n%s",
            date('Y-m-d H:i:s'),
            get_class($exception),
            $exception->getMessage(),
            $exception->getFile(),
            $exception->getLine(),
            $exception->getTraceAsString()
        );
        
        self::logError($message);
        self::showErrorPage();
        exit(1);
    }
    
    /**
     * Captura erros fatais (incluindo erro 500)
     */
    public static function handleFatalError() {
        $error = error_get_last();
        
        if ($error !== null && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            $message = sprintf(
                "[%s] [FATAL] %s in %s on line %d",
                date('Y-m-d H:i:s'),
                $error['message'],
                $error['file'],
                $error['line']
            );
            
            // Adicionar informações do contexto
            $context = [
                'URL' => isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : 'N/A',
                'Method' => isset($_SERVER['REQUEST_METHOD']) ? $_SERVER['REQUEST_METHOD'] : 'N/A',
                'IP' => self::getClientIp(),
                'User Agent' => isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : 'N/A',
                'POST Data' => !empty($_POST) ? json_encode($_POST, JSON_UNESCAPED_UNICODE) : 'Empty',
                'GET Data' => !empty($_GET) ? json_encode($_GET, JSON_UNESCAPED_UNICODE) : 'Empty',
                'Session Data' => isset($_SESSION) ? json_encode($_SESSION, JSON_UNESCAPED_UNICODE) : 'No session'
            ];
            
            $message .= "\nContext:\n" . print_r($context, true);
            $message .= "\n" . str_repeat("=", 80) . "\n";
            
            self::logError($message);
            
            // Limpar qualquer saída anterior
            if (ob_get_length()) {
                ob_clean();
            }
            
            self::showErrorPage();
        }
    }
    
    /**
     * Registra erro no arquivo de log
     */
    private static function logError($message) {
        $logMessage = $message . "\n";
        
        // Adicionar separador visual
        $logMessage .= str_repeat("-", 80) . "\n";
        
        // Escrever no arquivo de log
        error_log($logMessage, 3, self::$logFile);
        
        // Rotacionar log se ficar muito grande (>10MB)
        if (file_exists(self::$logFile) && filesize(self::$logFile) > 10485760) {
            $backupFile = self::$logDir . '/error_' . date('Y-m-d_H-i-s') . '.log';
            rename(self::$logFile, $backupFile);
        }
    }
    
    /**
     * Obtém o IP real do cliente
     */
    private static function getClientIp() {
        $ipKeys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 
                   'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR'];
        
        foreach ($ipKeys as $key) {
            if (isset($_SERVER[$key]) && filter_var($_SERVER[$key], FILTER_VALIDATE_IP)) {
                return $_SERVER[$key];
            }
        }
        
        return 'UNKNOWN';
    }
    
    /**
     * Exibe página de erro amigável para o usuário
     */
    private static function showErrorPage() {
        http_response_code(500);
        
        // Se já enviamos headers, não podemos mais
        if (headers_sent()) {
            echo '<h1>Erro interno do servidor</h1>';
            return;
        }
        
        ?>
        <!DOCTYPE html>
        <html lang="pt-BR">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>Erro - TRDAP</title>
            <style>
                * {
                    margin: 0;
                    padding: 0;
                    box-sizing: border-box;
                }
                body {
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    min-height: 100vh;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    padding: 20px;
                }
                .error-container {
                    background: white;
                    border-radius: 12px;
                    box-shadow: 0 10px 40px rgba(0,0,0,0.2);
                    max-width: 600px;
                    width: 100%;
                    padding: 40px;
                    text-align: center;
                }
                .error-icon {
                    font-size: 72px;
                    margin-bottom: 20px;
                }
                h1 {
                    color: #333;
                    font-size: 28px;
                    margin-bottom: 15px;
                }
                p {
                    color: #666;
                    font-size: 16px;
                    line-height: 1.6;
                    margin-bottom: 25px;
                }
                .error-code {
                    background: #f5f5f5;
                    border-left: 4px solid #764ba2;
                    padding: 15px;
                    margin: 20px 0;
                    text-align: left;
                    font-family: 'Courier New', monospace;
                    font-size: 14px;
                    color: #333;
                }
                .btn-home {
                    display: inline-block;
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    color: white;
                    padding: 12px 30px;
                    border-radius: 6px;
                    text-decoration: none;
                    font-weight: 600;
                    transition: transform 0.2s, box-shadow 0.2s;
                }
                .btn-home:hover {
                    transform: translateY(-2px);
                    box-shadow: 0 5px 20px rgba(118, 75, 162, 0.4);
                }
                .support-info {
                    margin-top: 30px;
                    padding-top: 30px;
                    border-top: 1px solid #eee;
                    font-size: 14px;
                    color: #999;
                }
            </style>
        </head>
        <body>
            <div class="error-container">
                <div class="error-icon">⚠️</div>
                <h1>Ops! Algo deu errado</h1>
                <p>Desculpe, encontramos um erro interno no servidor. Nossa equipe foi notificada automaticamente e está trabalhando para resolver o problema.</p>
                
                <div class="error-code">
                    <strong>Código do Erro:</strong> 500 - Internal Server Error<br>
                    <strong>Data/Hora:</strong> <?php echo date('d/m/Y H:i:s'); ?><br>
                    <strong>Referência:</strong> <?php echo uniqid('ERR-'); ?>
                </div>
                
                <a href="/" class="btn-home">← Voltar para a Página Inicial</a>
                
                <div class="support-info">
                    Se o problema persistir, entre em contato com o suporte.
                </div>
            </div>
        </body>
        </html>
        <?php
    }
    
    /**
     * Log personalizado para debug
     */
    public static function customLog($message, $context = []) {
        $logMessage = sprintf(
            "[%s] [CUSTOM] %s",
            date('Y-m-d H:i:s'),
            $message
        );
        
        if (!empty($context)) {
            $logMessage .= "\nContext: " . json_encode($context, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        }
        
        self::logError($logMessage);
    }
}

